# Copyright 2015 Sean Vig
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from __future__ import annotations

import os
import xml.etree.ElementTree as ET
from dataclasses import dataclass

from .copyright import Copyright, copyright_default
from .description import Description
from .element import Element
from .interface import Interface
from .printer import Printer


@dataclass(frozen=True)
class Protocol(Element):
    """Protocol scanner object

    Main scanner object that acts on the input xml files to generate protocol
    files.

    Required attributes: `name`

    Child elements: `copyright?`, `description?`, and `interface+`

    :param input_file: Name of input XML file
    """

    name: str
    copyright: Copyright | None
    description: Description | None
    interface: list[Interface]

    @classmethod
    def parse_file(cls, input_file: str) -> Protocol:
        if not os.path.exists(input_file):
            raise ValueError(f"Input xml file does not exist: {input_file}")
        xmlroot = ET.parse(input_file).getroot()
        if xmlroot.tag != "protocol":
            raise ValueError(
                f"Input file not a valid Wayland protocol file: {input_file}"
            )

        return cls.parse(xmlroot)

    @classmethod
    def parse(cls, element: ET.Element) -> Protocol:
        return cls(
            name=cls.parse_attribute(element, "name"),
            copyright=cls.parse_optional_child(element, Copyright, "copyright"),
            description=cls.parse_optional_child(element, Description, "description"),
            interface=cls.parse_repeated_child(element, Interface, "interface"),
        )

    def __repr__(self) -> str:
        return f"Protocol({self.name})"

    def output(self, output_dir: str, module_imports: dict[str, str]) -> None:
        """Output the scanned files to the given directory

        :param output_dir: Path of directory to output protocol files to
        :type output_dir: string
        """
        protocol_name = self.name.replace("-", "_")

        output_dir = os.path.join(output_dir, protocol_name)
        if not os.path.exists(output_dir):
            os.mkdir(output_dir)

        # First, we'll create the __init__.py file
        printer = Printer(protocol_name)
        if self.copyright:
            self.copyright.output(printer)
        else:
            printer(copyright_default)

        printer()
        for iface in sorted(self.interface, key=lambda x: x.name):
            printer(f"from .{iface.name} import {iface.class_name}  # noqa: F401")

        init_path = os.path.join(output_dir, "__init__.py")
        with open(init_path, "wb") as f:
            printer.write(f)

        # Now build all the modules
        for iface in self.interface:
            module_path = os.path.join(output_dir, iface.name + ".py")

            printer = Printer(self.name.replace("-", "_"), iface.name, module_imports)
            if self.copyright:
                self.copyright.output(printer)
            else:
                printer(copyright_default)
            printer()

            iface.output(printer, module_imports)

            with open(module_path, "wb") as f:
                printer.write(f)
